package de.juhu.distributor;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import de.juhu.util.Config;

/**
 * Diese Klasse bildet einen Schler ab und beinhaltet alle diesbezglich
 * wichtigen Informationen.
 * 
 * @version 1.1
 * @category Distribution
 * @author Juhu1705
 * @implNote Ist {@link Comparable vergleichbar} mit anderen {@link Student
 *           Schlern}.
 */
public class Student implements Comparable<Student>, Serializable {

	// INFO: Attribute

	/**
	 * Speichert alle vom {@link Student Schler} gewhlten {@link Course Kurse}
	 * nach ihrer Prioritt. An Position 0 steht der Kurs mit der geringsten
	 * Prioritt.
	 */
	private ArrayList<Course> courses = new ArrayList<>();

	/**
	 * Der Kurs, in dem sich der Schler momentan befindet. Ist {@code null}, wenn
	 * der Schler in keinem Kurs ist.
	 */
	private Course activeCourse;

	/**
	 * <p>
	 * Die Prioritt die dem {@link #activeCourse aktiven Kurs} nach der
	 * {@link #courses Liste der Kurse} zugeordnet ist.
	 * </p>
	 * Kann ber {@link #refreshPriority()} aktualisiert werden.
	 */
	protected int priority;

	/**
	 * Die ID des Schlers. Sie ist fr jeden unterschiedlichen Schler einzigartig.
	 */
	private int id;

	/**
	 * Der Nachname des Schlers.
	 */
	private String name;

	/**
	 * Der Vorname des Schlers.
	 */
	private String prename;

	/**
	 * Ein Indikator, der anzeigt, ob dieser Schler einem Kurs zugeordnet ist.
	 */
	private boolean mark;

	// INFO: Konstruktoren

	/**
	 * Erzeugt einen Schler mit einer neuen ID, der noch keinen Namen besitzt.
	 */
	protected Student() {
		this.generateID();
		this.unmark();
	}

	/**
	 * Erzeugt einen neuen Schler, mit einer neuen ID. Der Name und der Vorname
	 * werden gespeichert und die mitgegebenen Kurse werden in gleicher Reihenfolge
	 * in die {@link #courses Liste der gewnschten Kurse} gespeichert. Der erste
	 * Kurs der Liste wird dabei zum {@link #activeCourse aktiven Kurs} des
	 * Schlers.
	 * 
	 * @param name    Der Nachname des Schlers
	 * @param prename Der Vorname des Schlers
	 * @param courses Die Kurswnsche des Schlers
	 */
	public Student(String name, String prename, Course... courses) {
		this();
		for (Course courseName : courses) {
			this.addCourse(courseName);
		}

		if (this.courses.size() == 1 && this.courses.get(0).equals(Distributor.getInstance().ignore()))
			this.activeCourse = this.courses.get(0);

		this.setName(name);
		this.setPrename(prename);
		this.priority = 0;
	}

	/**
	 * Erstellt einen Schler mit dem mitgegebenen Vor- und Nachnamen, sowie der
	 * mitgegebenen ID. Nur zum Kopieren verwendet. Kurse mssen manuell eingefgt,
	 * oder beim kopieren synchronisiert werden. Hierzu eignet sich die Methode
	 * {@link Distributor#copyData(ArrayList, ArrayList, Course)}
	 * 
	 * @param name    Der Nachname des Schlers
	 * @param prename Der Vorname des Schlers
	 * @param id      Die ID des Schlers
	 */
	private Student(String name, String prename, int id) {
		this.setName(name);
		this.setPrename(prename);
		this.priority = 0;
		this.id = id;
		this.unmark();
	}

	// INFO: Methoden

	/**
	 * Generiert eine ID fr den Schler. Wird immer beim erstellen eines Schlers
	 * aufgerufen. Greift auf die Methode {@link Distributor#getStudentID()} zurck
	 * um eine ID zu generieren und setzt die ID auf den von dieser Methode
	 * zurckgegebenen Wert.
	 */
	private void generateID() {
		this.id = Distributor.getStudentID();
	}

	/**
	 * <p>
	 * Setzt den {@link #activeCourse aktive Kurs} auf den nchsten noch freien
	 * {@link Course Kurs} aus der {@link #courses Liste der gewnschten Kurse} des
	 * Schlers.
	 * </p>
	 * <ul>
	 * <li>Dazu wird zunchst geguckt, ob der Schler {@link #courses gewnschten
	 * Kurse} besitzt. Sollte dies nicht der Fall sein, so wird {@code false}
	 * zurckgegeben, der Priorittswert durch Aufruf der Methode
	 * {@link #refreshPriority()} aktualisiert und die Methode beendet.</li>
	 * 
	 * <li>Danach wird geschaut, ob der {@link #activeCourse aktive Kurs} momentan
	 * den Wert {@code null} beinhaltet. Sollte dies der Fall sein, so wird der
	 * aktive Kurs, auf den ersten Kurs aus der Liste der {@link #courses
	 * gewnschten Kurse} gesetzt und der Schler dem Kurs hinzugefgt.</li>
	 * 
	 * <li>Im folgenden wird in einer {@code while}-Schleife geschaut, ob der aktive
	 * Kurs bereits mit seiner maximalen Schleranzahl besetzt ist. Wenn ja, so wird
	 * der nchste Kurs aus der {@link #courses Liste der gewnschten Kurse}
	 * ermittelt und bei diesem das gleiche ermittelt. Sollten alle Kurse die der
	 * Schler sich wnscht an dieser Stelle voll sein, so wird die Methode mit dem
	 * Rckgabewert {@code false} abgebrochen, sowie der {@link #activeCourse aktive
	 * Kurs} auf {@code null} gesetzt.</li>
	 * 
	 * <li>Sollte ein Kurs gefunden werden, der noch frei ist, so wird dieser
	 * Schler der {@link Course#students Liste der Schler des Kurses} hinzugefgt.
	 * Anschlieend wird der {@link #priority Priorittswert} des Schlers durch die
	 * Methode {@link #refreshPriority()} aktualisiert. Dann wird die Methode mit
	 * der Rckgabe {@code true} beendet.</li>
	 * </ul>
	 * 
	 * @return Ob ein neuer Kurs zugewiesen werden konnte. Wenn {@code false}
	 *         zurckgegeben wird, so ist der {@link #activeCourse aktive Kurs} dann
	 *         {@code null}.
	 */
	public boolean next() {
		if (this.courses.isEmpty())
			return false;

		if (this.activeCourse == null) {
			this.activeCourse = this.courses.get(0);
			this.activeCourse.addStudent(this);
		}

		while (this.activeCourse.isFull()) {
			this.setNextCourse();
			if (this.activeCourse == null) {
				this.refreshPriority();
				return false;
			}
		}

		this.refreshPriority();
		return true;
	}

	/**
	 * Setzt den {@link #activeCourse Aktiven Kurs} auf den nchsten Kurs aus der
	 * {@link #courses Liste der gewnschten Kurse} des Schlers.
	 * 
	 * @see #next() Hier ist die Funktion genau beschrieben. Anstatt der
	 *      {@code while}-Schleife wird hier lediglich ein {@code if}-Case genutzt,
	 *      der mit einer {@code else} Bedingung mit der vorgeschalteten berprfung
	 *      verbunden ist. Ansonsten sind die Methoden identisch aufgebaut.
	 * @implNote Die Methode fgt einen Schler auch zu einem Bereits vollen Kurs
	 *           hinzu und lsst ihn dort verweilen.
	 * @return Ob der Schler einem Kurs zugeordnet werden konnte.
	 */
	public boolean onlyNext() {
		if (this.courses.isEmpty())
			return false;

		if (this.activeCourse == null) {
			this.activeCourse = this.courses.get(0);
			this.activeCourse.addStudent(this);
		} else if (this.activeCourse.isFull()) {
			this.setNextCourse();
			if (this.activeCourse == null) {
				this.refreshPriority();
				return false;
			}
		}

		this.refreshPriority();
		return true;
	}

	// INFO: Getter und Setter

	/**
	 * Setzt den {@link #name Nachnamen} des Schlers auf den mitgegebenen String.
	 * 
	 * @param name Den Nachnamen, den der Schler annehmen soll.
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Gibt den {@link #name Nachnamen} des Schlers zurck.
	 * 
	 * @return Der Nachname des Schlers.
	 */
	public String getName() {
		return this.name;
	}

	/**
	 * Setzt den {@link #prename Vornamen} des Schlers auf den mitgegebenen String.
	 * 
	 * @param prename Den Vornamen, den der Schler annehmen soll.
	 */
	public void setPrename(String prename) {
		this.prename = prename;
	}

	/**
	 * Gibt den {@link #prename Vornamen} des Schlers zurck.
	 * 
	 * @return Der Vorname des Schlers.
	 */
	public String getPrename() {
		return this.prename;
	}

	/**
	 * Fgt einen Kurs ans Ende der {@link #courses Liste der gewnschten Kurse} des
	 * Schlers hinzu.
	 * 
	 * Wenn der Kurs den Wert {@code null} darstellt, so wird dieser Ignoriert.
	 * 
	 * @param course Der Kurs, der hinzugefgt werden soll.
	 */
	public void addCourse(Course course) {
		if (course == null)
			return;
		if (this.courses.contains(course)) {
			this.courses.remove(course);
		}

		this.courses.add(course);
	}

	/**
	 * Fgt einen Kurs an der gewnschten Stelle zu den {@link #courses
	 * Wunschkursen} hinzu. Dabei werden die nachfolgenden Kurse und der Kurs an
	 * diesem Index um einen Listenplatz nach hinten verschoben.
	 * 
	 * @param index  Der Listenplatz, an den der Kurs eingefgt werden soll.
	 * @param course Der Kurs, der eingefgt werden soll.
	 * @throws IndexOutOfBoundsException Wenn der Index nicht innerhalb der Liste
	 *                                   liegt.
	 */
	public void addCourse(int index, Course course) throws IndexOutOfBoundsException {
		if (course == null)
			return;
		if (this.courses.contains(course)) {
			this.courses.remove(course);
		}

		this.courses.add(index, course);
	}

	/**
	 * Gibt die Liste aller {@link #courses gewnschten Kurse} des Schlers zurck.
	 * 
	 * @return Ein Array der gewnschten Kurse des Schlers.
	 */
	public Course[] getCourses() {
		return this.courses.toArray(new Course[this.courses.size()]);
	}

	/**
	 * Setzt den {@link #activeCourse aktiven Kurs} auf den nchsten Kurs aus der
	 * {@link #courses Liste der gewnschten Kurse}.
	 * 
	 * @return Der neu gesetzte {@link #activeCourse Kurs}.
	 */
	private Course setNextCourse() {
		return this.setCourse(this.getNextCourse());
	}

	/**
	 * Setzt den {@link #activeCourse aktiven Kurs} auf den gewnschten Kurs. Dabei
	 * wird der letzte aktive Kurs in einen entfernt.
	 * 
	 * @param course Der Kurs der zum aktiven Kurs werden soll.
	 * @return Der aktive Kurs.
	 */
	private Course setCourse(Course course) {
		if (this.activeCourse.contains(this))
			this.activeCourse.removeStudent(this);

		if (course == null)
			return this.activeCourse = null;

		this.activeCourse = course;
		this.activeCourse.addStudent(this);

		return course;
	}

	/**
	 * Gibt denjenigen Kurs aus der {@link #courses Liste der gewnschten Kurse des
	 * Schlers} zurck, welcher dem {@link #activeCourse aktiven Kurs} folgt.
	 * 
	 * @return Der dem aktiven Kurs folgende Kurs.
	 * @see #getNextCourse(Course, int)
	 *      getNextCourse({@link #activeCourse}{@link #getNextCourse(Course, int) ,
	 *      1)}
	 */
	public Course getNextCourse() {
		return this.getNextCourse(this.activeCourse, 1);
	}

	/**
	 * Gibt den Kurs zurck der iterator viele Positionen hinter dem
	 * {@link #activeCourse aktiven Kurs} in der {@link #courses Liste der
	 * gewnschten Kurse} steht.
	 * 
	 * @param iterator Anzahl der Positionen die zwischen dem aktiven Kurs und dem
	 *                 zu ermittelnden Kurs liegen sollen.
	 * @return Den iterator-viele Positionen hinter dem aktiven Kurs liegende Kurs.
	 * @see #getNextCourse(Course, int)
	 *      getNextCourse({@link #activeCourse}{@link #getNextCourse(Course, int) ,
	 *      iterator)}
	 */
	public Course getNextCourse(int iterator) {
		return this.getNextCourse(this.activeCourse, iterator);
	}

	/**
	 * Ermittelt den Kurs, der iterator viele Positionen hinter dem mitgegebenen
	 * Kurs in der {@link #courses Liste der gewnschten Kurse} liegt.
	 * 
	 * @param course   Der Kurs, von dem gezhlt wird.
	 * @param iterator Die Anzahl an Positionen, die vom mitgegebenen Kurs
	 *                 weitergegangen werden soll.
	 * @return Der Kurs der entsprechend viele Positionen hinter dem mitgegebenen
	 *         Kurs liegt. Gibt {@code null} zurck wenn der Kurs kein Element der
	 *         {@link #courses Liste der gewnschten Kurse} ist. Gibt den ersten
	 *         Kurs der Liste zurck wenn der mitgegebene Kurs {@code null} ist.
	 */
	public Course getNextCourse(Course course, int iterator) {
		if (course == null)
			return this.courses.get(0);
		if (this.courses.contains(course)) {
			int i = this.courses.indexOf(course);
			if (i + iterator >= this.courses.size())
				return null;
			else
				return this.courses.get(i + iterator);
		}
		return null;
	}

	/**
	 * Ermittelt den Index an dem der Kurs in der {@link #courses Liste der
	 * gewnschten Kurse} steht multipliziert mit 2.
	 * 
	 * @param course Der Kurs dessen Wert ermittelt werden soll
	 * @return Der Index multipliziert mit 2, oder {@link Integer#MAX_VALUE}, wenn
	 *         der Kurs nicht in der {@link #courses Liste der gewnschten Kurse}
	 *         existiert.
	 */
	public int getCourseAmount(Course course) {
		for (int i = 0; i < this.courses.size(); i++) {
			if (this.courses.get(i).equals(course)) {
				// References.LOGGER.info(this.toString() + ": " + i);
				return i * 2;
			}
		}
		return Integer.MAX_VALUE;
	}

	/**
	 * @return Den {@link #activeCourse aktiven Kurs} des Schlers
	 */
	public Course getActiveCourse() {
		return activeCourse;
	}

	/**
	 * Setzt den aktiven Kurs des Schlers auf den gewnschten Kurs, wenn dieser in
	 * der {@link #courses Liste der Wunschkurse} vorhanden ist. Danach sorgt die
	 * Methode dafr, das die {@link #priority Prioritt} des Kurses aktualisiert
	 * wird.
	 * 
	 * @param activeCourse Der Kurs, welcher zum aktiven Kurs gemacht werden soll.
	 *                     Wenn er {@code null} entspricht wird der
	 *                     {@link #activeCourse aktive Kurs} auf {@link null}
	 *                     gesetzt.
	 */
	public void setActiveCourse(Course activeCourse) {
		if (this.isMarked())
			this.unmark();

		if (activeCourse == null) {
			if (this.activeCourse != null && this.activeCourse.contains(this))
				this.activeCourse.removeStudent(this);
			this.activeCourse = activeCourse;
			this.refreshPriority();
			return;
		}

		if (this.courses.contains(activeCourse)) {
			if (this.isMarked())
				this.mark = false;
			if (this.activeCourse != null && this.activeCourse.contains(this))
				this.activeCourse.removeStudent(this);
			this.activeCourse = activeCourse;
			if (!this.activeCourse.contains(this))
				this.activeCourse.addStudent(this);
			this.refreshPriority();
		}
	}

	/**
	 * Setzt den {@link #priority Priorittswert} auf den, durch die Methode
	 * {@link #calculatePriority()} ermittelten Wert.
	 */
	public void refreshPriority() {
		this.priority = this.calculatePriority();
	}

	/**
	 * Berechnet die Prioritt die der {@link #activeCourse aktive Kurs} besitzt.
	 * 
	 * @return Die Prioritt die der aktive Kurs besitzt, oder
	 *         {@link Integer#MAX_VALUE} wenn der aktive Kurs {@code null} ist, oder
	 *         nicht in der {@link #courses Liste der Wunschkurse} existiert.
	 */
	private int calculatePriority() {
		if (this.activeCourse == null)
			return Integer.MAX_VALUE;
		if (this.courses.contains(this.activeCourse))
			return (this.courses.indexOf(this.activeCourse) + 1);

		return Integer.MAX_VALUE;
	}

	/**
	 * @return Gibt die zuletzt ermittelte {@link #priority Prioritt} zurck.
	 */
	public int getPriority() {
		return this.priority;
	}

	/**
	 * Ermittelt die Rate, die dieser Schler besitzt: {@link #priority Prioritt}
	 * hoch {@link Config#powValue}. Sollte die Prioritt {@link Integer#MAX_VALUE}
	 * entsprechen, wird die durch
	 * {@link Distributor#getHighestPriorityWhithoutIntegerMax()} ermittelte
	 * Prioritt plus eins als Prioritt angenommen.
	 * 
	 * @return Die Rate des Schlers
	 */
	public int getRate() {
		return (int) (this.priority == Integer.MAX_VALUE
				? Math.pow(Distributor.getInstance().getHighestPriorityWhithoutIntegerMax() + 1, Config.powValue)
				: Math.pow(this.priority, Config.powValue));
	}

	/**
	 * Ermittelt die Rate, die dieser Schler besitzt: {@link #priority Prioritt}
	 * hoch {@link Config#powValue}. Sollte die Prioritt {@link Integer#MAX_VALUE}
	 * entsprechen, wird die durch highestPriority plus eins ersetzt.
	 * 
	 * @param highestPriority Die hchste Prioritt der Berechnung
	 * @return Die Rate des Schlers
	 */
	public int getRate(int highestPriority) {
		return (int) (this.priority == Integer.MAX_VALUE ? Math.pow(highestPriority + 1, Config.powValue + 3)
				: Math.pow(this.priority, Config.powValue));
	}

	/**
	 * Setzt die {@link #mark Markierung} des Schlers auf {@code true}.
	 */
	public void mark() {
		this.mark = true;
	}

	/**
	 * Setzt die {@link #mark Markierung} des Schlers auf {@code false}.
	 */
	public void unmark() {
		this.mark = false;
	}

	/**
	 * @return Die {@link #mark Markierung} des Schlers.
	 */
	public boolean isMarked() {
		return this.mark;
	}

	@Override
	public int compareTo(Student s) {
		int i = this.getName().compareToIgnoreCase(s.getName());

		if (i == 0)
			i = this.getPrename().compareToIgnoreCase(s.getPrename());

		return i;
	}

	/**
	 * Schaut ob das zu vergleichende Objekt ein Schler ist, wenn nicht wird der
	 * Rckgabewert der {@link Object#equals(Object) super-Methode} zurckgegeben.
	 * Zum vergleichen der beiden Schler wird die {@link #id ID} der Schler
	 * miteinander verglichen.
	 */
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof Student))
			return false;

		Student student = (Student) obj;

		return student.id == this.id;

	}

	/**
	 * Gibt als representativen String den {@link #prename Vornamen} verbunden mit
	 * dem {@link #name Nachnamen} zurck.
	 */
	@Override
	public String toString() {
		return this.prename + " " + this.name;
	}

	@Override
	protected Object clone() throws CloneNotSupportedException {
		Student s = new Student(this.name, this.prename, this.id);
		return s;
	}

	/**
	 * Der Index des Kurses in der {@link #courses Liste der Wunschkurse}
	 * 
	 * @see {@link ArrayList#indexOf(Object)}
	 */
	public int getPosition(Course c) {
		return this.courses.indexOf(c);
	}

	/**
	 * berprft, ob die ID mit der ID dieses Schlers bereinstimmt.
	 * 
	 * @param studentID Die ID, deren bereinstimmung geprft wird.
	 * @return Ob die ID mit der {@link #id ID} dieses Schlers bereinstimmt.
	 */
	public boolean idequals(int studentID) {
		return this.id == studentID;
	}

	/**
	 * @return Die ID dieses Schlers
	 */
	public int getID() {
		return this.id;
	}

	/**
	 * Fgt die mitgegebene Kurse in gleicher Reihenfolge ans Ende der
	 * {@link #courses Liste der Wunschkurse} des Schlers an.
	 * 
	 * @param c Die Kurse sie eingefgt werden sollen.
	 */
	public void setCourses(Course... c) {
		this.courses.clear();

		for (Course course : c)
			this.addCourse(course);
	}

	/**
	 * @return Die {@link #courses Wunschkurse} dieses Schlers.
	 */
	public List<Course> getCoursesAsList() {
		return this.courses;
	}

	/**
	 * Aktualisiert die Marke des Schlers in Bezug auf dessen {@link #priority
	 * Prioritt}. Ist die {@link #priority Prioritt} des Schlers kleiner 0, oder
	 * grer als die highestPriority wird der Schler {@link #mark() markiert}, ist
	 * der Schler {@link #mark markiert}, obwohl die vornestehenden Bedingungen
	 * nicht erfllt sind, wird die {@link #unmark() markierung zurckgezogen}.
	 * 
	 * @param highestPriority Die hchste Prioritt, die in dem zu vergleichenden
	 *                        Prozess vorliegt.
	 */
	public void checkMarkt(int highestPriority) {

		if (this.getPriority() > highestPriority || this.getPriority() < 0)
			this.mark();
		else if (this.mark)
			this.unmark();
	}
}
