package de.juhu.distributor;

import java.io.Serializable;
import java.util.ArrayList;

import de.juhu.util.Config;

/**
 * Diese Klasse bildet einen Kurs ab und beinhaltet alle diesbezglich wichtigen
 * Informationen.
 * 
 * @version 1.0
 * @category Distribution
 * @author Juhu1705
 * @implNote Ist {@link Comparable vergleichbar} mit anderen {@link Course
 *           Kursen}.
 */
public class Course implements Comparable<Course>, Serializable {

	/**
	 * Alle Schler, die dem Kurs zugeordnet sind.
	 */
	private ArrayList<Student> students = new ArrayList<>();

	/**
	 * Der Lehrer des Kurses / Das Lehrerkrzel
	 */
	private String teacher;

	/**
	 * Das Fach des Kurses
	 */
	private String subject;

	/**
	 * Die Maximale Schleranzahl, die dieser Kurs beinhalten darf.
	 */
	private int maxStudents;

	/**
	 * Erzeugt einen Kurs, indem das maximale Schlerlimit, der Lehrer und das Fach
	 * gesetzt sind.
	 * 
	 * @param subject     Das Fach des Kurses
	 * @param teacher     Der Lehrer des Kurses
	 * @param maxStudents Die maximale Anzahl an Schlern, die in diesen Kurs passen
	 */
	public Course(String subject, String teacher, int maxStudents) {
		this(subject, teacher);

		this.maxStudents = maxStudents;
	}

	/**
	 * Erzeugt einen Kurs, in dem der Lehrer und das Fach mitgegeben werden. Die
	 * maximale Schlerzahl wird auf den in der Config mitgegebenen Standartwert
	 * gesetzt.
	 * 
	 * @param subject Das Fach des Kurses
	 * @param teacher Der Schler des Kurses
	 */
	public Course(String subject, String teacher) {
		this.teacher = teacher.toUpperCase();
		this.subject = subject.toUpperCase();
		this.maxStudents = Config.normalStudentLimit;
	}

	/**
	 * Erzeugt einen Kurs aus einem zwei langen Array, indem der Kurs und der Lehrer
	 * gesetzt wird und die maximale Schlerzahl auf den in der Config mitgegebenen
	 * Standartwert gesetzt wird.
	 * 
	 * @param split An Position 0 das Fach und an Position 1 der Lehrer
	 * @throws NullPointerException Wenn das gegebene Array weniger als zwei Strings
	 *                              beinhaltet.
	 */
	public Course(String[] split) throws NullPointerException {
		this(split[0], split[1]);
	}

	/**
	 * @return Der Lehrer des Kurses / Das Lehrerkrzel
	 */
	public String getTeacher() {
		return this.teacher;
	}

	/**
	 * @return Das Fach des Kurses
	 */
	public String getSubject() {
		return this.subject;
	}

	/*
	 * TODO Bitte berprfen
	 */

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof String) {
			return ((String) obj).equals(this.toString());
		}
		if (obj instanceof Course) {
			return ((Course) obj).toString().equals(this.toString());
		}

		return super.equals(obj);
	}

	/**
	 * berprft, ob mehr Schler im Kurs sind, als das {@link #maxStudents
	 * Schlerlimit} zulsst. Ist das {@link #maxStudents Schlerlimit} {@code -1}
	 * besitzt der Kurs kein Schlerlimit.
	 * 
	 * @return {@code true}, wenn mehr Schler im Kurs sind, als das Schlerlimit
	 *         erlaubt.
	 */
	public boolean isFull() {
		return ((this.maxStudents < this.students.size()) && (this.maxStudents != -1));
	}

	/**
	 * Gibt den Kurs als String zurck. Dazu wird das {@link #subject Fach} durch
	 * einen "|" vom {@link #teacher Lehrer} getrennt. Der zurckgegebene String
	 * sieht dann wiefolgt aus: {@link #subject
	 * Fach}{@code  + "|" + }{@link #teacher Lehrer}.
	 */
	@Override
	public String toString() {
		return this.subject + "|" + this.teacher;
	}

	/**
	 * @return Gibt eine Liste aller Schler im Kurs zurck.
	 */
	public ArrayList<Student> getStudents() {
		return this.students;
	}

	/**
	 * berprft, ob dieser Schler diesem Kurs zugeordnet ist.
	 * 
	 * @param student Der Schler, bei dem berprft werden soll, ob er im Kurs ist.
	 * @return Ob der Schler im Kurs ist.
	 * @implNote Verwendet die Methode {@link ArrayList#contains(Object)}
	 */
	public boolean contains(Student student) {
		return this.students.contains(student);
	}

	/**
	 * Fgt einen Schler hinzu, wenn er nicht bereits im Kurs ist, oder der
	 * hinzuzufgene Schler nicht {@code null} ist.
	 * 
	 * @param student Der Schler, der hinzugefgt werden soll.
	 * @return Ob der Schler erfolgreich hinzugefgt werden konnte.
	 * @implNote Verwendet die Methode {@link ArrayList#add(Object)}
	 */
	public boolean addStudent(Student student) {
		if (student == null || this.contains(student))
			return false;

		return this.students.add(student);
	}

	/**
	 * Entfernt einen Schler aus diesem Kurs.
	 * 
	 * @param student Der Schler, der entfernt werden soll.
	 * @return Der Schler, der entfernt wurde.
	 * @implNote Verwendet die Methode {@link ArrayList#remove(int)}
	 */
	public Student removeStudent(Student student) {
		return this.students.remove(this.students.indexOf(student));
	}

	/**
	 * Gibt den Schler zurck, der dem mitgegebene Index in der {@link #students
	 * Liste der Schler des Kurses} zugeordnet ist.
	 * 
	 * @param i Der Index von dem der Schler zurckgegeben werden soll.
	 * @return Der Schler, der an der Position i der Liste steht.
	 */
	public Student getStudent(int i) {
		return this.students.get(i);
	}

	/**
	 * @return {@link ArrayList#size()} der {@link #students Liste der Schler des
	 *         Kurses}.
	 */
	public int size() {
		return this.students.size();
	}

	/**
	 * Gibt die Liste als String zurck. Die Schler werden ber
	 * {@link Student#toString()} in einen String berfhrt.
	 * 
	 * @return {@link ArrayList#toString()} der {@link #students Liste der Schler
	 *         des Kurses}.
	 */
	public String studentsToString() {
		return this.students.toString();
	}

	@Override
	public int compareTo(Course c) {
		int i = this.subject.compareTo(c.subject);

		if (i == 0)
			i = this.teacher.compareTo(c.teacher);

		return 0;
	}

	@Override
	protected Object clone() throws CloneNotSupportedException {
		return new Course(subject, teacher, maxStudents);
	}

	/**
	 * @return Die maximale Anzahl an Schlern, die diesem Kurs zugewiesen werden
	 *         drfen.
	 */
	public int getMaxStudentCount() {
		return this.maxStudents;
	}

	/**
	 * Setzt den {@link #teacher Lehrer} des Kurses.
	 * 
	 * @param teacher Der Lehrer des Kurses.
	 */
	public void setTeacher(String teacher) {
		this.teacher = teacher.toUpperCase();
	}

	/**
	 * Setzt das {@link #subject Fach} des Kurses.
	 * 
	 * @param subject Das Fach des Kurses.
	 */
	public void setSubject(String subject) {
		this.subject = subject.toUpperCase();
	}

	/**
	 * Setzt das {@link #maxStudents Schlerlimit} des Kurses
	 * 
	 * @param maxStudents Die maximale Anzahl an Schlern, die diesem Kurs
	 *                    zugeordnet werden drfen.
	 */
	public void setStudentMax(int maxStudents) {
		this.maxStudents = maxStudents;
	}

}
